/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.util;

import java.util.MissingResourceException;
import java.util.TreeMap;
import java.util.logging.Level;

import com.zynaptic.reaction.Logger;

/**
 * Provides a basic log service which prints all log messages generated by the
 * reactor to the console.
 * 
 * @author Chris Holgate
 */
public final class ReactorLogSystemOut implements ReactorLogTarget {

  // Map used to store individual logger instances.
  private final TreeMap<String, Logger> loggerMap = new TreeMap<String, Logger>();

  /*
   * Implements ReactorLogTarget.getLogger(...)
   */
  public final synchronized Logger getLogger(String loggerId,
      String loggerResources) throws MissingResourceException {

    // TODO: Localisation is not currently supported.
    if (loggerResources != null)
      throw new MissingResourceException(
          "The basic console log target does not support localisation.", null,
          loggerResources);

    // Create new logger instances on request.
    Logger logger = loggerMap.get(loggerId);
    if (logger == null) {
      logger = new LoggerSystemOut(loggerId);
      loggerMap.put(loggerId, logger);
    }
    return logger;
  }

  /**
   * Provides a basic logger implementation which simply prints the required log
   * information to the console.
   */
  private final class LoggerSystemOut implements Logger {

    // Internal logger state.
    private final String loggerId;
    private Level logLevel;

    /**
     * Implements the default constructor which is used to assign the unique ID
     * for this logger instance.
     * 
     * @param loggerId
     *          This is the unique ID which is associated with the logger
     *          instance.
     */
    private LoggerSystemOut(String loggerId) {
      this.loggerId = loggerId;
      this.logLevel = Level.CONFIG;
    }

    /*
     * Implements Logger.log(...)
     */
    public final synchronized void log(Level level, String message) {
      log(level, message, null);
    }

    /*
     * Implements Logger.log(...)
     */
    public final synchronized void log(Level level, String message,
        Throwable exception) {
      if (level.intValue() >= logLevel.intValue()) {
        System.out.println("[" + loggerId + "] " + level + " : " + message);
        if (exception != null) {
          System.out.print("[" + loggerId + "] " + "THROWN : ");
          exception.printStackTrace(System.out);
        }
      }
    }

    /*
     * Implements Logger.getLoggerId()
     */
    public final synchronized String getLoggerId() {
      return loggerId;
    }

    /*
     * Implements Logger.getLogLevel()
     */
    public final synchronized Level getLogLevel() {
      return logLevel;
    }

    /*
     * Implements Logger.setLogLevel(...)
     */
    public final synchronized void setLogLevel(Level logLevel) {
      this.logLevel = logLevel;
    }
  }
}