/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.util;

import java.util.MissingResourceException;
import java.util.TreeMap;
import java.util.logging.Level;

import com.zynaptic.reaction.Logger;

/**
 * Provides a basic log service which routes all log messages generated by the
 * reactor to the standard Java log service.
 * 
 * @author Chris Holgate
 */
public final class ReactorLogJavaStandard implements ReactorLogTarget {

  // Map used to store individual logger instances.
  private final TreeMap<String, LoggerJavaStandard> loggerMap = new TreeMap<String, LoggerJavaStandard>();

  /*
   * Implements ReactorLogTarget.getLogger(...)
   */
  public synchronized final Logger getLogger(String loggerId,
      String loggerResources) throws MissingResourceException {
    LoggerJavaStandard logger = loggerMap.get(loggerId);

    // Create new logger if one is not already present.
    if (logger == null) {
      logger = new LoggerJavaStandard();
      loggerMap.put(loggerId, logger);
    }

    // Refresh the standard Java logger reference with the new parameters.
    logger.refreshJavaLogger(loggerId, loggerResources);
    return logger;
  }

  /**
   * Provides a basic logger implementation which simply routes the required log
   * information to the standard Java logger.
   */
  private final class LoggerJavaStandard implements Logger {

    // The wrapped standard Java logger component.
    private java.util.logging.Logger javaLogger;

    /*
     * Refreshes the reference to the standard Java logger, updating the logger
     * resources reference if required.
     */
    protected final synchronized void refreshJavaLogger(String loggerId,
        String loggerResources) {
      javaLogger = java.util.logging.Logger
          .getLogger(loggerId, loggerResources);
    }

    /*
     * Implements Logger.log(...)
     */
    public final synchronized void log(Level level, String msg) {
      StackTraceElement callerContext = Thread.currentThread().getStackTrace()[2];
      javaLogger.logp(level, callerContext.getClassName(),
          callerContext.getMethodName(), msg);
    }

    /*
     * Implements Logger.log(...)
     */
    public final synchronized void log(Level level, String msg, Throwable thrown) {
      StackTraceElement callerContext = Thread.currentThread().getStackTrace()[2];
      javaLogger.logp(level, callerContext.getClassName(),
          callerContext.getMethodName(), msg, thrown);
    }

    /*
     * Implements Logger.getLoggerId()
     */
    public final synchronized String getLoggerId() {
      return javaLogger.getName();
    }

    /*
     * Implements Logger.getLogLevel()
     */
    public final synchronized Level getLogLevel() {
      return javaLogger.getLevel();
    }

    /*
     * Implements Logger.setLogLevel(...)
     */
    public final synchronized void setLogLevel(Level logLevel) {
      javaLogger.setLevel(logLevel);
    }
  }
}
