/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.util;

/**
 * Implements a monotonic clock source derived from the standard Java wallclock.
 * This monotonic clock variant uses the standard Java millisecond clock which
 * is susceptible to changes made to the underlying wallclock. It works around
 * potential changes to the wallclock time by maintaining a clock offset which
 * is updated when the wallclock is observed to go backwards or jump too far
 * into the future.
 * <p>
 * To minimise timing errors when such discontinuities occur, a sleepy thread
 * makes regular calls to {@link #getMsTime() getMsTime}, since timed calls to
 * the thread <code>sleep</code> method are assumed to be independent of the
 * wallclock time. This places an upper limit on the interval between requests
 * to {@link #getMsTime() getMsTime}.
 * 
 * @author Chris Holgate
 * 
 */
public final class FixedUpMonotonicClock implements MonotonicClockSource {

  // This sets the maximum interval at which getMsTime() should be called.
  private static final int MAX_MS_INTERVAL = 1000;

  // Used to track the timer offset and the last timer sampling point.
  private long timeOffset, timeLastRead;

  /*
   * Implements MonotonicClockSource.getMsTime()
   */
  public final synchronized long getMsTime() {
    long timeCurrent = System.currentTimeMillis() - timeOffset;
    long timeDelta = timeCurrent - timeLastRead;

    // Fix up clock if it has gone backwards.
    if (timeDelta < 0) {
      timeCurrent = timeLastRead;
      timeOffset += timeDelta;
    }

    // Fix up clock if it has jumped forwards.
    else if (timeDelta > MAX_MS_INTERVAL * 2) {
      timeCurrent = timeLastRead + MAX_MS_INTERVAL;
      timeOffset += timeDelta - MAX_MS_INTERVAL;
    }

    // Fixed up version of the current time.
    timeLastRead = timeCurrent;
    return timeLastRead;
  }

  /*
   * Implements MonotonicClockSource.init()
   */
  public final synchronized void init() {
    timeLastRead = 0;
    timeOffset = System.currentTimeMillis();
    Thread thread = new Thread(new PeriodicRequestGenerator());
    thread.start();
  }

  /*
   * Implements periodic requests to the fixed up monotonic clock in order to
   * detect step changes in the underlying clock source.
   */
  private final class PeriodicRequestGenerator implements Runnable {
    public final void run() {
      while (!Thread.currentThread().isInterrupted()) {
        getMsTime();
        try {
          Thread.sleep(MAX_MS_INTERVAL);
        } catch (InterruptedException error) {
          // Handled using isInterrupted()
        }
      }
    }
  }
}
