/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.osgi;

import java.util.MissingResourceException;
import java.util.TreeMap;
import java.util.logging.Level;

import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.service.log.LogService;
import org.osgi.util.tracker.ServiceTracker;

import com.zynaptic.reaction.Logger;
import com.zynaptic.reaction.Reactor;
import com.zynaptic.reaction.util.ReactorLogTarget;

/**
 * Provides support for logging to an optional OSGi logging service.
 * Availability of a suitable OSGi logging bundle is checked on startup and if
 * present this will be used for the entire lifecycle of the Reaction bundle. A
 * fallback logging target may be specified on startup which will be used if a
 * suitable OSGi service is not available.
 * 
 * @author Chris Holgate
 */
public final class ReactorLogOsgiOptional implements ReactorLogTarget {

  // Map used to store individual logger instances.
  private final TreeMap<String, Logger> loggerMap = new TreeMap<String, Logger>();

  // Backup logging target.
  private final ReactorLogTarget backupLogTarget;

  // Local handle on the OSGI logging service tracker.
  private final ServiceTracker<LogService, LogService> logServiceTracker;

  // Reference back to reaction service.
  private final ServiceReference<Reactor> reactorServiceReference;

  /**
   * Implements default constructor for the optional OSGi logging service.
   * Attaches to the OSGi log service if available and falls back to the backup
   * logging target if not.
   * 
   * @param bundleContext
   *          This is the OSGi bundle context associated with the Reaction
   *          bundle.
   * @param backupLogTarget
   *          This is the backup logging target which will be used if the
   *          optional OSGi log service is not available.
   */
  public ReactorLogOsgiOptional(BundleContext bundleContext,
      ReactorLogTarget backupLogTarget) {
    this.backupLogTarget = backupLogTarget;
    this.logServiceTracker = logServiceTrackerInit(bundleContext);
    this.reactorServiceReference = bundleContext
        .getServiceReference(Reactor.class);
  }

  /*
   * Initialisation function for OSGi log service tracker.
   */
  private final synchronized ServiceTracker<LogService, LogService> logServiceTrackerInit(
      BundleContext bundleContext) {
    ServiceTracker<LogService, LogService> tracker = null;
    try {
      tracker = new ServiceTracker<LogService, LogService>(bundleContext,
          LogService.class, null);
      tracker.open();
      backupLogTarget.getLogger("com.zynaptic.reaction.osgi", null).log(
          Level.INFO,
          "OSGi logging service installed - backup service will not be used.");
    } catch (NoClassDefFoundError exc) {
      if (backupLogTarget != null) {
        backupLogTarget.getLogger("com.zynaptic.reaction.osgi", null).log(
            Level.WARNING,
            "OSGi logging service not available - using backup log service.",
            exc);
      }
    }
    return tracker;
  }

  /*
   * Implements ReactorLogTarget.getLogger(...)
   */
  public final synchronized Logger getLogger(String loggerId,
      String loggerResources) throws MissingResourceException {

    // TODO: Localisation is not currently supported.
    if (loggerResources != null)
      throw new MissingResourceException(
          "The basic OSGi log target does not support localisation.", null,
          loggerResources);

    // Create new logger instances on request.
    Logger logger = loggerMap.get(loggerId);
    if (logger == null) {
      logger = new LoggerBasicOsgi(loggerId, loggerResources);
      loggerMap.put(loggerId, logger);
    }
    return logger;
  }

  /**
   * Provides a OSGi logger implementation which maps the Java log service
   * levels to the OSGi log service equivalents.
   */
  private final class LoggerBasicOsgi implements Logger {

    // Internal logger state.
    private final String loggerId;
    private final String loggerResources;
    private Level logLevel;
    private Logger backupLogger = null;

    /**
     * Implements the default constructor which is used to assign the unique ID
     * for this logger instance.
     * 
     * @param loggerId
     *          This is the unique ID which is associated with the logger
     *          instance.
     */
    private LoggerBasicOsgi(String loggerId, String loggerResources) {
      this.loggerId = loggerId;
      this.loggerResources = loggerResources;
      this.logLevel = Level.CONFIG;
    }

    /*
     * Implements ReactorLogTarget.log(...)
     */
    public final synchronized void log(Level level, String message) {
      boolean logged = false;
      if (logServiceTracker != null) {
        LogService logService = logServiceTracker.getService();
        if (logService != null) {
          if (level.intValue() >= logLevel.intValue())
            logService.log(reactorServiceReference, mapLogLevel(level), "["
                + loggerId + "]" + message);
          logged = true;
        }
      }

      // Fallback to backup log service.
      if (!logged && (backupLogTarget != null)) {
        if (backupLogger == null)
          backupLogger = backupLogTarget.getLogger(loggerId, loggerResources);
        backupLogger.log(level, message);
      }
    }

    /*
     * Implements ReactorLogTarget.log(...)
     */
    public final synchronized void log(Level level, String message,
        Throwable exception) {
      boolean logged = false;
      if (logServiceTracker != null) {
        LogService logService = logServiceTracker.getService();
        if (logService != null) {
          if (level.intValue() >= logLevel.intValue())
            logService.log(reactorServiceReference, mapLogLevel(level), "["
                + loggerId + "] " + message, exception);
          logged = true;
        }
      }

      // Fallback to backup log service.
      if (!logged && (backupLogTarget != null)) {
        if (backupLogger == null)
          backupLogger = backupLogTarget.getLogger(loggerId, loggerResources);
        backupLogger.log(level, message, exception);
      }
    }

    /*
     * Implements Logger.getLoggerId()
     */
    public final synchronized String getLoggerId() {
      return loggerId;
    }

    /*
     * Implements Logger.getLogLevel()
     */
    public final synchronized Level getLogLevel() {
      return logLevel;
    }

    /*
     * Implements Logger.setLogLevel(...)
     */
    public final synchronized void setLogLevel(Level logLevel) {
      this.logLevel = logLevel;

      // Update log level on backup log target.
      if (backupLogTarget != null) {
        if (backupLogger == null)
          backupLogger = backupLogTarget.getLogger(loggerId, loggerResources);
        backupLogger.setLogLevel(logLevel);
      }
    }

    /*
     * Map from standard Java log level to OSGi integer log level values.
     */
    private final int mapLogLevel(Level level) {
      int osgiLevel;
      if (level.intValue() >= Level.SEVERE.intValue())
        osgiLevel = LogService.LOG_ERROR;
      else if (level.intValue() >= Level.WARNING.intValue())
        osgiLevel = LogService.LOG_WARNING;
      else if (level.intValue() >= Level.CONFIG.intValue())
        osgiLevel = LogService.LOG_INFO;
      else
        osgiLevel = LogService.LOG_DEBUG;
      return osgiLevel;
    }
  }
}