/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.osgi;

import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceRegistration;

import com.zynaptic.reaction.Reactor;
import com.zynaptic.reaction.core.ReactorCore;
import com.zynaptic.reaction.util.FixedUpMonotonicClock;
import com.zynaptic.reaction.util.MonotonicClockSource;
import com.zynaptic.reaction.util.ReactorLogSystemOut;
import com.zynaptic.reaction.util.ReactorLogTarget;

/**
 * Implements default bundle activation for the Reaction asynchronous
 * programming framework.
 * 
 * @author Chris Holgate
 */
public final class ReactionDefaultActivator implements BundleActivator {

  // Reactor user API service registration.
  private ServiceRegistration<Reactor> reactorService;

  /**
   * Implements the <code> BundleActivator.start </code> method which is used to
   * start up the reactor on activation.
   * 
   * @param context
   *          The OSGi bundle context.
   */
  public final void start(BundleContext context) {

    // This hardcodes the source of the reactor clock.
    MonotonicClockSource reactorClock = new FixedUpMonotonicClock();

    // Use optional OSGi log service if available, with fallback logging to the
    // console.
    ReactorLogTarget logService = new ReactorLogOsgiOptional(context,
        new ReactorLogSystemOut());

    // Start up the reactor.
    ReactorCore.getReactorControl().start(reactorClock, logService);

    // Register the reactor user interface as a provided service.
    reactorService = context.registerService(Reactor.class,
        ReactorCore.getReactor(), null);
  }

  /**
   * Implements the <code> BundleActivator.stop </code> method which is used to
   * stop the reactor on deactivation.
   * 
   * @param context
   *          The OSGi bundle context.
   * @throws Exception
   *           An Exception is thrown if the current thread is interrupted while
   *           waiting for the reactor to stop. An Error is thrown if the
   *           reactor was stopped due to the corresponding error condition.
   */
  public final void stop(BundleContext context) throws Exception {

    // Stop the reactor.
    ReactorCore.getReactorControl().stop();
    ReactorCore.getReactorControl().join();

    // Disable external access to the reactor service.
    reactorService.unregister();
    reactorService = null;
  }

}
