/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.core;

import java.util.HashSet;
import java.util.Iterator;

import com.zynaptic.reaction.ReactorNotRunningException;
import com.zynaptic.reaction.RestrictedCapabilityException;
import com.zynaptic.reaction.Signal;
import com.zynaptic.reaction.SignalContextException;
import com.zynaptic.reaction.Signalable;

/**
 * This class provides the implementation of the {@link Signal} API.
 * 
 * @author Chris Holgate
 */
public final class SignalCore<T> implements Signal<T> {

  // Local handle on the reactor.
  private final ReactorCore reactorCore = ReactorCore.getReactorCore();

  // Set of signalable objects associated with this signal.
  private final HashSet<Signalable<T>> signalableSet = new HashSet<Signalable<T>>();

  // Flag used to prevent addition and removal of signalable objects during
  // callbacks.
  private boolean callbacksRunning = false;

  /*
   * Implements Signal.subscribeSignalable(...)
   */
  public final synchronized void subscribe(final Signalable<T> signalable)
      throws SignalContextException {
    if (callbacksRunning == false) {
      if (!signalableSet.contains(signalable)) {
        signalableSet.add(signalable);
      }
    } else {
      throw new SignalContextException(
          "Attempted to subscribe signalable from signal callback.");
    }
  }

  /*
   * Implements Signal.unsubscribeSignalable(...)
   */
  public final synchronized void unsubscribe(final Signalable<T> signalable)
      throws SignalContextException {
    if (callbacksRunning == false) {
      signalableSet.remove(signalable);
    } else {
      throw new SignalContextException(
          "Attempted to unsubscribe signalable from signal callback.");
    }
  }

  /*
   * Delegates Signal.signal(...)
   */
  public final void signal(final T data) throws ReactorNotRunningException {
    reactorCore.signal(this, data, false);
  }

  /*
   * Delegates Signal.signalFinalize(...)
   */
  public final void signalFinalize(final T data)
      throws ReactorNotRunningException {
    reactorCore.signal(this, data, true);
  }

  /*
   * Implements Signal.makeRestricted()
   */
  public final Signal<T> makeRestricted() {
    return new RestrictedSignalCore(this);
  }

  /*
   * Provide equivalence testing between normal and restricted signal object
   * references.
   */
  @Override
  public final boolean equals(Object signal) {
    if (signal instanceof SignalCore.RestrictedSignalCore)
      return signal.equals(this);
    else
      return super.equals(signal);
  }

  /*
   * Processes a signal event by forwarding the specified data to all subscribed
   * signalable objects.
   */
  protected final synchronized void processSignal(final T data,
      final boolean finalize) {
    callbacksRunning = true;

    // Issue callbacks to all subscribed signalable objects.
    Iterator<Signalable<T>> sigIter = signalableSet.iterator();
    while (sigIter.hasNext()) {
      sigIter.next().onSignal(this.makeRestricted(), data);
    }

    // Remove all signalable objects when finalized.
    if (finalize) {
      signalableSet.clear();
    }
    callbacksRunning = false;
  }

  /*
   * Implement restricted interface wrapper for signal objects.
   */
  private final class RestrictedSignalCore implements Signal<T> {
    private final SignalCore<T> signalCore;

    public RestrictedSignalCore(SignalCore<T> signalCore) {
      this.signalCore = signalCore;
    }

    public final void subscribe(Signalable<T> signalable) {
      signalCore.subscribe(signalable);
    }

    public final void unsubscribe(Signalable<T> signalable) {
      signalCore.unsubscribe(signalable);
    }

    public final void signal(T data) throws RestrictedCapabilityException {
      throw new RestrictedCapabilityException(
          "Method not available for restricted Signal interface.");
    }

    public final void signalFinalize(T data)
        throws RestrictedCapabilityException {
      throw new RestrictedCapabilityException(
          "Method not available for restricted Signal interface.");
    }

    public final Signal<T> makeRestricted() {
      return this;
    }

    @Override
    public final boolean equals(Object signal) {
      return signalCore.equals(signal);
    }

    @Override
    public final int hashCode() {
      return signalCore.hashCode();
    }
  }
}
