/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.core;

import java.util.LinkedList;

import com.zynaptic.reaction.Deferrable;
import com.zynaptic.reaction.Deferred;
import com.zynaptic.reaction.DeferredTerminationException;
import com.zynaptic.reaction.DeferredTimedOutException;
import com.zynaptic.reaction.DeferredTriggeredException;
import com.zynaptic.reaction.ReactorNotRunningException;
import com.zynaptic.reaction.RestrictedCapabilityException;
import com.zynaptic.reaction.Timeable;

/**
 * This class provides the core implementation of the {@link Deferred} API.
 * Generic type checking is carried out on the public interface, so processing
 * within the scope of this core implementation can make use of unchecked type
 * references.
 * 
 * @author Chris Holgate
 */
public final class DeferredCore<T> implements Deferred<T>, Timeable<T> {

  // Deferred state flags.
  private boolean callbackDataValid = false;
  private boolean callbackErrorValid = false;
  private boolean callbackChainTerminated = false;
  private boolean ignoreCallback = false;

  // Stack context used for debugging deferred event objects when they go out of
  // scope without triggering.
  private final StackTraceElement[] stackTraceElements;

  // Callback chain data.
  private Object callbackData = null;
  private Exception callbackError = null;
  private final LinkedList<Deferrable<?, ?>> callbackChain = new LinkedList<Deferrable<?, ?>>();

  // Local handle on the reactor.
  private final ReactorCore reactorCore = ReactorCore.getReactorCore();

  /**
   * The default constructor sets the stack context under which the deferred
   * event was created. This allows exceptions due to untriggered deferred
   * events going out of scope to return meaningful debug information.
   */
  public DeferredCore() {
    stackTraceElements = Thread.currentThread().getStackTrace();
  }

  /**
   * Tidy up the deferred event object when it is garbage collected. This adds
   * warnings to the logs if the deferred goes out of scope without being
   * triggered or terminated.
   */
  @Override
  public final void finalize() {
    if ((!callbackDataValid) && (!callbackErrorValid)) {
      callbackError = new DeferredTimedOutException(
          "Deferred : going out of scope (untriggered).");
      callbackError.setStackTrace(stackTraceElements);
      reactorCore.closeDeferred(this, callbackError);
    } else if (!callbackChainTerminated) {
      callbackError = new DeferredTerminationException(
          "Deferred : going out of scope (unterminated).");
      callbackError.setStackTrace(stackTraceElements);
      reactorCore.closeDeferred(this, callbackError);
    }
  }

  /*
   * Implements Deferred.addDeferrable(...)
   */
  public final synchronized <U> Deferred<U> addDeferrable(
      final Deferrable<T, U> deferrable, final boolean terminal)
      throws DeferredTerminationException {

    // Check to see if the callback chain has been terminated.
    if (callbackChainTerminated) {
      throw new DeferredTerminationException(
          "Deferred : callback chain already terminated.");
    }

    // Append deferrable to the callback chain.
    callbackChain.addLast(deferrable);

    // Mark callback chain as terminated and process it if possible.
    if (terminal) {
      callbackChainTerminated = true;
      if ((callbackDataValid) || (callbackErrorValid)) {
        reactorCore.processDeferred(this);
      }
    }
    return this.typePunned();
  }

  /*
   * Implements Deferred.addDeferrable(...)
   */
  public final synchronized <U> Deferred<U> addDeferrable(
      final Deferrable<T, U> deferrable) throws DeferredTerminationException {

    // Check to see if the callback chain has been terminated.
    if (callbackChainTerminated) {
      throw new DeferredTerminationException(
          "Deferred : callback chain already terminated.");
    }

    // Append deferrable to the callback chain.
    callbackChain.addLast(deferrable);
    return this.typePunned();
  }

  /*
   * Implements Deferred.terminate()
   */
  public final synchronized Deferred<T> terminate() {

    // Check to see if the callback chain has been terminated.
    if (callbackChainTerminated) {
      throw new DeferredTerminationException(
          "Deferred : callback chain already terminated.");
    }

    // Mark callback chain as terminated and process it if possible.
    callbackChainTerminated = true;
    if ((callbackDataValid) || (callbackErrorValid)) {
      reactorCore.processDeferred(this);
    }
    return this;
  }

  /*
   * Implements Deferred.callback(...)
   */
  public final synchronized void callback(final T data)
      throws DeferredTriggeredException {

    // Discard the callback after a timeout.
    if (ignoreCallback) {
      ignoreCallback = false;
    }

    // Check to see if this deferred event has already been triggered.
    else if ((callbackDataValid) || (callbackErrorValid)) {
      throw new DeferredTriggeredException("Deferred : already triggered.");
    }

    // Cache the new data parameter and process the callback chain.
    else {
      callbackData = data;
      callbackDataValid = true;
      reactorCore.cancelTimer(this);
      if (callbackChainTerminated) {
        reactorCore.processDeferred(this);
      }
    }
  }

  /*
   * Implements Deferred.errback(...)
   */
  public final synchronized void errback(final Exception error)
      throws DeferredTriggeredException {

    // Discard the errback after a timeout.
    if (ignoreCallback) {
      ignoreCallback = false;
    }

    // Check to see if this deferred event has already been triggered.
    else if ((callbackDataValid) || (callbackErrorValid)) {
      throw new DeferredTriggeredException("Deferred : already triggered.");
    }

    // Cache the new error parameter and process the callback chain.
    else {
      callbackError = error;
      callbackErrorValid = true;
      reactorCore.cancelTimer(this);
      if (callbackChainTerminated) {
        reactorCore.processDeferred(this);
      }
    }
  }

  /*
   * Implements Deferred.setTimeout(...)
   */
  public final Deferred<T> setTimeout(final int msTimeout)
      throws ReactorNotRunningException {
    reactorCore.runTimerOneShot(this, msTimeout, null);
    return this;
  }

  /*
   * Implements Deferred.cancelTimeout(...)
   */
  public final Deferred<T> cancelTimeout() {
    reactorCore.cancelTimer(this);
    return this;
  }

  /*
   * Implements Deferred.discard()
   */
  public final void discard() {
    this.terminate();
  }

  /*
   * Implements Deferred.makeRestricted()
   */
  public final Deferred<T> makeRestricted() {
    return new RestrictedDeferredCore(this);
  }

  /*
   * Provide equivalence testing between normal and restricted deferred object
   * references.
   */
  @Override
  public final boolean equals(Object deferred) {
    if (deferred instanceof DeferredCore.RestrictedDeferredCore)
      return deferred.equals(this);
    else
      return super.equals(deferred);
  }

  /*
   * Implements Timeable.tick(...)
   */
  public final synchronized void onTick(final T data) {

    // If not already triggered, queue the timeout errback and discard
    // future callback and errback requests.
    if ((!callbackDataValid) && (!callbackErrorValid)) {
      ignoreCallback = true;
      callbackError = new DeferredTimedOutException("Timeout in deferred.");
      callbackErrorValid = true;
      if (callbackChainTerminated) {
        reactorCore.processDeferred(this);
      }
    }
  }

  /*
   * Process callback chain, calling all queued callbacks in sequence. This will
   * only be triggered once a terminal deferrable has been added and an event is
   * ready for processing. Note that deferrable generic types will have been
   * discarded before the callback chain is processed, so raw deferrable
   * references are used within the scope of this function.
   */
  @SuppressWarnings({ "unchecked", "rawtypes" })
  protected final synchronized void processCallbackChain(
      final boolean reactorRunning) {

    // Deal with callbacks if reactor is not running.
    if (!reactorRunning) {
      callbackError = new ReactorNotRunningException(
          "Can't schedule deferred callbacks when reactor not running.");
      callbackErrorValid = true;
    }

    // Propagate the callbacks.
    while (!callbackChain.isEmpty()) {
      Deferrable deferrable = callbackChain.removeFirst();
      try {

        // Issue error callback for pending errors.
        if (callbackErrorValid) {
          callbackData = deferrable.onErrback(this.makeRestricted(),
              callbackError);
        }

        // Issue standard callback for pending data.
        else if (callbackDataValid) {
          callbackData = deferrable.onCallback(this.makeRestricted(),
              callbackData);
        }
        callbackErrorValid = false;
        callbackDataValid = true;
      }

      // Exceptions thrown during callback processing are stored for forwarding
      // to the next deferrable in the list. Errors are left to propagate back
      // to the reactor.
      catch (Exception err) {
        callbackError = err;
        callbackErrorValid = true;
        callbackDataValid = false;
      }
    }

    // Any data returned by the terminal deferrable is discarded.
    // Terminal error conditions are passed to the reactor on closure.
    reactorCore.closeDeferred(this, callbackErrorValid ? callbackError : null);
  }

  /*
   * Provide type punning function for reassigning the generic type of a given
   * deferred event object.
   */
  @SuppressWarnings("unchecked")
  protected final <PT> Deferred<PT> typePunned() {
    return (Deferred<PT>) this;
  }

  /*
   * Implement restricted interface wrapper for deferred objects.
   */
  private final class RestrictedDeferredCore implements Deferred<T> {
    private final DeferredCore<T> deferredCore;

    public RestrictedDeferredCore(DeferredCore<T> deferredCore) {
      this.deferredCore = deferredCore;
    }

    public final void callback(T data) throws RestrictedCapabilityException {
      throw new RestrictedCapabilityException(
          "Method not available for restricted Deferred interface.");
    }

    public final void errback(Exception error)
        throws RestrictedCapabilityException {
      throw new RestrictedCapabilityException(
          "Method not available for restricted Deferred interface.");
    }

    public final <U> Deferred<U> addDeferrable(Deferrable<T, U> deferrable,
        boolean terminal) throws DeferredTerminationException {
      return deferredCore.addDeferrable(deferrable, terminal);
    }

    public final <U> Deferred<U> addDeferrable(Deferrable<T, U> deferrable)
        throws DeferredTerminationException {
      return deferredCore.addDeferrable(deferrable);
    }

    public final Deferred<T> terminate() throws DeferredTerminationException {
      return deferredCore.terminate();
    }

    public final Deferred<T> setTimeout(int msTimeout)
        throws ReactorNotRunningException {
      deferredCore.setTimeout(msTimeout);
      return this;
    }

    public final Deferred<T> cancelTimeout() {
      deferredCore.cancelTimeout();
      return this;
    }

    public final void discard() {
      deferredCore.discard();
    }

    public final Deferred<T> makeRestricted() {
      return this;
    }

    @Override
    public final boolean equals(Object deferred) {
      return deferredCore.equals(deferred);
    }

    @Override
    public final int hashCode() {
      return deferredCore.hashCode();
    }
  }
}
