/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.core;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import com.zynaptic.reaction.Deferrable;
import com.zynaptic.reaction.Deferred;
import com.zynaptic.reaction.DeferredConcentrator;
import com.zynaptic.reaction.DeferredTerminationException;

/**
 * This class provides the concrete implementation of the
 * {@link DeferredConcentrator} API.
 * 
 * @author Chris Holgate
 * 
 */
public final class DeferredConcentratorCore<T> implements
    DeferredConcentrator<T>, Deferrable<T, T> {

  // The deferred event object used for output.
  private Deferred<List<T>> deferredOutput = null;

  // Map used to associate input deferred events to result array positions.
  private int resultIndex = 0;
  private final HashMap<Deferred<T>, Integer> resultIndexMap = new HashMap<Deferred<T>, Integer>();

  // Cache the input callback data.
  private Exception callbackError = null;
  private final ArrayList<T> callbackData = new ArrayList<T>();

  // Local handle on the reactor.
  private final ReactorCore reactorCore = ReactorCore.getReactorCore();

  /*
   * Attach multiple input events to the deferred list.
   */
  public final synchronized void addInputDeferred(final Deferred<T> deferred)
      throws DeferredTerminationException {

    // Check to see if the output deferred has already been requested.
    if (deferredOutput != null) {
      throw new DeferredTerminationException(
          "No more deferred concentrator inputs can be added.");
    }

    // Add the deferred input to the result index map.
    resultIndexMap.put(deferred, new Integer(resultIndex++));

    // Terminate the deferred input callback chain.
    deferred.addDeferrable(this, true);
  }

  /*
   * Get a handle on the output deferred event object.
   */
  public final synchronized Deferred<List<T>> getOutputDeferred() {

    // Create the deferred output event if required. Issue the callbacks if
    // ready.
    if (deferredOutput == null) {
      callbackData.trimToSize();
      deferredOutput = reactorCore.newDeferred();
      if (callbackError != null) {
        deferredOutput.errback(callbackError);
      } else if (resultIndexMap.size() == 0) {
        deferredOutput.callback(callbackData);
      }
    }
    return deferredOutput.makeRestricted();
  }

  /*
   * Handle callbacks from input callback chains.
   */
  public final synchronized T onCallback(final Deferred<T> deferred,
      final T data) {

    // Retrieve the result index from the map.
    int resultIndex = (resultIndexMap.remove(deferred)).intValue();

    // Extend the array if required.
    while (callbackData.size() <= resultIndex) {
      callbackData.add(null);
    }

    // Insert the result into the results array.
    callbackData.set(resultIndex, data);

    // Generate the output callback if all input callbacks have completed.
    if ((deferredOutput != null) && (resultIndexMap.size() == 0)) {
      deferredOutput.callback(callbackData);
    }
    return null;
  }

  /*
   * Handle errbacks from input callback chains.
   */
  public final synchronized T onErrback(final Deferred<T> deferred,
      final Exception error) {
    if (callbackError == null) {
      callbackError = error;
      if (deferredOutput != null) {
        deferredOutput.errback(callbackError);
      }
    }
    return null;
  }
}
