/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction;

/**
 * Defines the interface to a threadable task object. Threadable task objects
 * can be submitted to the reactor in order to run in a separate thread. This is
 * useful for executing code which contains blocking calls within the Reaction
 * framework. On submitting a threadable task object to the reactor using the
 * {@link Reactor#runThread(Threadable, Object) runThread} method on the
 * {@link Reactor} interface, an object implementing the {@link Deferred}
 * interface is returned which will be used to forward the results generated by
 * the {@link #run(Object) run} method on completion.
 * 
 * @param <T>
 *          This type parameter specifies the type of data object which may be
 *          passed into the threadable task.
 * @param <U>
 *          This type parameter specifies the type of data object which will be
 *          passed back by the threadable task on completion.
 * 
 * @author Chris Holgate
 */
public interface Threadable<T, U> {

  /**
   * Runs a threadable task in the context of a new thread. On submitting an
   * object implementing this interface for threaded execution, this method will
   * be called by the reactor in the context of a new thread. On normal
   * completion it will return a data object which will be passed as the
   * callback parameter to the deferred callback chain. Alternatively, if an
   * exception is thrown this will be passed as the errback parameter to the
   * error callback chain.
   * 
   * @param data
   *          This is the input data object which is provided when submitting
   *          the threadable task object to the reactor.
   * @return Returns an output data object which will be passed to the deferred
   *         callback chain as the
   *         {@link Deferrable#onCallback(Deferred, Object) onCallback}
   *         parameter.
   * @throws Exception
   *           On an exception, the exception object is passed to the deferred
   *           error callback chain as the
   *           {@link Deferrable#onErrback(Deferred, Exception) onErrback}
   *           parameter.
   */
  public U run(T data) throws Exception;

}