/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction;

/**
 * Defines the signal event interface. This provides the user API for
 * manipulating signal event objects. Signal event objects can only be created
 * by using the {@link Reactor#newSignal() newSignal} method on the
 * {@link Reactor} interface.
 * 
 * @param <T>
 *          This type parameter specifies the type of the data object which will
 *          be passed each time the signal is triggered.
 * 
 * @author Chris Holgate
 */
public interface Signal<T> {

  /**
   * Subscribes a new signalable object. This method is used in order to
   * subscribe a signalable object to this signal, which means that the
   * signalable object will be notified of any events associated with the
   * signal.
   * 
   * @param signalable
   *          This is the signalable object which is being subscribed.
   * @throws SignalContextException
   *           This runtime exception is thrown if there is an attempt to
   *           subscribe a new signalable object from the context of a signal
   *           event callback.
   * @throws ReactorNotRunningException
   *           This exception is thrown when attempt is made to subscribe to a
   *           signal when the reactor is not running.
   */
  public void subscribe(Signalable<T> signalable)
      throws SignalContextException, ReactorNotRunningException;

  /**
   * Unsubscribes a signalable object. This method is used in order to
   * unsubscribe a signalable object so that it no longer receives event
   * notifications associated with the signal.
   * 
   * @param signalable
   *          This is the signalable object which is being unsubscribed.
   * @throws SignalContextException
   *           This runtime exception is thrown if there is an attempt to
   *           unsubscribe a signalable object from the context of a signal
   *           event callback.
   * @throws ReactorNotRunningException
   *           This exception is thrown when attempt is made to unsubscribe from
   *           a signal when the reactor is not running.
   */
  public void unsubscribe(Signalable<T> signalable)
      throws SignalContextException, ReactorNotRunningException;

  /**
   * Sends a signal event notification. This method is used in order to trigger
   * the signal event. When called the event will be broadcast to all subscribed
   * signalable objects.
   * 
   * @param data
   *          This is the signal data parameter which will be passed to each
   *          subscribed signalable object in turn. It is good practice to make
   *          this parameter object immutable.
   * @throws RestrictedCapabilityException
   *           This exception is thrown if an attempt is made to call this
   *           method on a signal event object reference with restricted
   *           capability.
   * @throws ReactorNotRunningException
   *           This exception is thrown if an attempt is made to signal an event
   *           when the reactor is not running.
   */
  public void signal(final T data) throws RestrictedCapabilityException,
      ReactorNotRunningException;

  /**
   * Sends a finalising signal event notification. Finalising signals are
   * broadcast to all subscribers in the same way as for a conventional signal
   * event, after which all subscribers are automatically unsubscribed from the
   * signal. This will typically be used to notify shutdown of a service.
   * 
   * @param data
   *          This is the signal data parameter which will be passed to each
   *          subscribed signalable object in turn. It is good practice to make
   *          this parameter object immutable.
   * @throws RestrictedCapabilityException
   *           This exception is thrown if an attempt is made to call this
   *           method on a signal event object reference with restricted
   *           capability.
   * @throws ReactorNotRunningException
   *           This exception is thrown if an attempt is made to signal an event
   *           when the reactor is not running.
   */
  public void signalFinalize(final T data)
      throws RestrictedCapabilityException, ReactorNotRunningException;

  /**
   * Convert the signal interface to restricted capability. This method should
   * be used to restrict the capability of a signal interface so that the
   * {@link #signal(Object) signal} and {@link #signalFinalize(Object)
   * signalFinalize} methods are protected from unauthorised use.
   * 
   * @return Returns a version of the current {@link Signal} interface with
   *         restricted capability.
   */
  public Signal<T> makeRestricted();

}