/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction;

import java.util.logging.Level;

/**
 * Provides a common logging interface which may be used by Reaction clients to
 * perform runtime message logging to a common backend log service. Note that
 * the log levels correspond to the standard Java logging library levels.
 * 
 * @author Chris Holgate
 */
public interface Logger {

  /**
   * Logs a message string at the specified severity level.
   * 
   * @param level
   *          This is the severity level at which the message should be logged.
   * @param msg
   *          This is the message string which should be included in the log
   *          file.
   */
  public void log(Level level, String msg);

  /**
   * Logs message string and accompanying exception information at the specified
   * severity level.
   * 
   * @param level
   *          This is the severity level at which the message should be logged.
   * @param msg
   *          This is the message string which should be included in the log
   *          file.
   * @param thrown
   *          This is the exception which should be logged with the accompanying
   *          message.
   */
  public void log(Level level, String msg, Throwable thrown);

  /**
   * Gets the unique logging identifier associated with a given logger instance.
   * 
   * @return Returns the unique logging identifier for the logger instance.
   */
  public String getLoggerId();

  /**
   * Gets the current logging level associated with a given logger instance.
   * 
   * @return Returns the current logging level for the logger instance.
   */
  public Level getLogLevel();

  /**
   * Sets the logging level to be used for a given logger instance. If the
   * underlying logging service supports hierarchical logging this will also set
   * the log level for all loggers below the current instance in the log
   * hierarchy.
   * 
   * @param logLevel
   *          This is the log level to be used by the logger instance for all
   *          future log messages.
   */
  public void setLogLevel(Level logLevel);

}
