/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction;

/**
 * Defines the deferrable callback interface. Any object which implements this
 * interface is referred to as a <em>deferrable object</em>, implying that it
 * supports deferred callbacks.
 * 
 * @param <T>
 *          This type identifier specifies the type of data which may be passed
 *          to the deferrable object via the
 *          {@link #onCallback(Deferred, Object) onCallback} method.
 * @param <U>
 *          This type identified specifies the type of data which will be
 *          returned by either the {@link #onCallback(Deferred, Object)
 *          onCallback} or {@link #onErrback(Deferred, Exception) onErrback}
 *          method.
 * @author Chris Holgate
 */
public interface Deferrable<T, U> {

  /**
   * This is the callback handler for deferrable objects. It is called in
   * response to a deferred event which indicates successful completion of a
   * given task.
   * 
   * @param deferred
   *          This is the deferred event object through which the callback was
   *          issued.
   * @param data
   *          This is a data object of type <code>T</code> which encapsulates
   *          any information being passed back as a result of the deferred
   *          event.
   * @return Returns a data object of type <code>U</code> which will be
   *         propagated to the next deferrable in the chain as the
   *         {@link #onCallback(Deferred, Object) onCallback} data parameter.
   * @throws Exception
   *           Exceptions thrown by callback handlers are propagated to the next
   *           deferrable in the chain as the
   *           {@link #onErrback(Deferred, Exception) onErrback} error
   *           parameter.
   */
  public U onCallback(Deferred<T> deferred, T data) throws Exception;

  /**
   * This is the error callback handler for deferrable objects. It is called in
   * response to a deferred error which indicates the failure of a given task.
   * 
   * @param deferred
   *          This is the deferred event object through which the error callback
   *          was issued.
   * @param error
   *          This is the Exception object which has been caught on task
   *          failure. It contains information about the cause of the error. It
   *          can be either handled locally or re-thrown to pass it up the
   *          callback chain.
   * @return Returns a data object of type <code>U</code> which will be
   *         propagated to the next deferrable in the chain as the
   *         {@link #onCallback(Deferred, Object) onCallback} data parameter.
   * @throws Exception
   *           Exceptions thrown by error callback handlers are propagated to
   *           the next deferrable in the chain as the
   *           {@link #onErrback(Deferred, Exception) onErrback} error
   *           parameter.
   */
  public U onErrback(Deferred<T> deferred, Exception error) throws Exception;

}
