/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.examples.thread;

import com.zynaptic.reaction.Deferrable;
import com.zynaptic.reaction.Deferred;
import com.zynaptic.reaction.Reactor;
import com.zynaptic.reaction.Threadable;
import com.zynaptic.reaction.core.ReactorControl;
import com.zynaptic.reaction.core.ReactorCore;
import com.zynaptic.reaction.util.FixedUpMonotonicClock;
import com.zynaptic.reaction.util.ReactorLogSystemOut;

/**
 * Basic example of deferred errback generation from a threadable task.
 */
public class ThreadableExample2 {
  public static void main(String[] args) throws InterruptedException {

    // Obtain a handle on the reactor control interface.
    reactorCtrl = ReactorCore.getReactorControl();
    reactor = ReactorCore.getReactor();

    // Start the reactor using the specified monotonic clock and log
    // service.
    System.out.println("Starting up reactor.");
    reactorCtrl.start(new FixedUpMonotonicClock(), new ReactorLogSystemOut());

    // Start up a long running task.
    Deferred<String> deferred = reactor.runThread(new LongRunningTask(), new String("Input Data"));

    // Attach callbacks to the long running task.
    deferred.addDeferrable(new SimpleDeferrable(), true);

    // Wait for reactor to exit.
    reactorCtrl.join();
    System.exit(0);
  }

  // Local handles on the reactor component.
  private static ReactorControl reactorCtrl = null;
  private static Reactor reactor = null;

  /**
   * Implementation of a simple deferrable object.
   */
  public static class SimpleDeferrable implements Deferrable<String, String> {

    // Callbacks are ignored in this example.
    public String onCallback(Deferred<String> deferred, String data) {
      return null;
    }

    // Errbacks are used to stop the reactor.
    public String onErrback(Deferred<String> deferred, Exception error) {
      System.out.println("Deferred error   : " + error);
      reactorCtrl.stop();
      return null;
    }
  }

  /**
   * Implementation of a long running task using a threadable.
   */
  public static class LongRunningTask implements Threadable<String, String> {
    public String run(String data) throws InterruptedException {
      System.out.println("Threadable input : " + data);
      Thread.sleep(1000);
      throw new InterruptedException("Task failed");
    }
  }
}
