/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.examples.swing;

import java.awt.Font;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.geom.Rectangle2D;
import java.text.DecimalFormat;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JPanel;

import com.zynaptic.reaction.Reactor;
import com.zynaptic.reaction.Signal;
import com.zynaptic.reaction.Signalable;
import com.zynaptic.reaction.Timeable;
import com.zynaptic.reaction.core.ReactorControl;
import com.zynaptic.reaction.core.ReactorCore;
import com.zynaptic.reaction.util.JavaMonotonicClock;
import com.zynaptic.reaction.util.ReactorLogSystemOut;

/**
 * Example of using the Reaction framework in conjunction with a GUI framework
 * (Swing) via the MVC pattern.
 * 
 * @author Chris Holgate
 */
public class SwingGuiExample {
  public static void main(String[] args) throws InterruptedException {

    // Start up the reactor.
    Reactor reactor = ReactorCore.getReactor();
    ReactorControl reactorCtrl = ReactorCore.getReactorControl();
    reactorCtrl.start(new JavaMonotonicClock(), new ReactorLogSystemOut());

    // Start up the application.
    ApplicationModel app = new ApplicationModel();
    app.startup(reactorCtrl, reactor);

    // Start up the GUI view and controller.
    ViewerController gui = new ViewerController();
    gui.startup(app);

    // Wait for the application to exit.
    reactorCtrl.join();
    System.exit(0);
  }

  /**
   * This is the interface which should be used for application control.
   */
  private static interface ApplicationControl {

    /**
     * Command interface used to start the stopwatch timer.
     */
    public void timerStart();

    /**
     * Command interface used to stop the stopwatch timer.
     */
    public void timerStop();

    /**
     * Command interface used to reset the stopwatch timer.
     */
    public void timerReset();

    /**
     * Command interface used to quit the stopwatch timer application.
     */
    public void timerQuit();

  }

  /**
   * This is the interface which should be used for viewer updates.
   */
  private static interface ApplicationView {

    /**
     * Get a handle on the timer update signal.
     */
    public Signal<Integer> getTimerUpdateSignal();

    /**
     * Read back the current timer value (in milliseconds).
     */
    public int getTimerData();

  }

  /**
   * Specify command IDs used to perform control actions.
   */
  enum CommandIds {
    TIMER_START, TIMER_STOP, TIMER_RESET, TIMER_QUIT
  };

  /**
   * Implement the application logic as a Reaction service.
   */
  private static class ApplicationModel implements Timeable<Integer>,
      Signalable<CommandIds>, ApplicationControl, ApplicationView {

    // Cached local reference to the reactor service.
    private Reactor reactor;
    private ReactorControl reactorCtrl;

    // Track the current timer state.
    private boolean timerRunning = false;
    private int timerValue = 0;

    // Local handle on the timer update signal.
    private Signal<Integer> timerUpdateSignal = null;
    private int timerUpdateSequence;

    // Local handle on the control interface signal.
    private Signal<CommandIds> controlInterfaceSignal = null;

    /*
     * Start up the application component once the Reaction service is
     * available.
     */
    public void startup(ReactorControl reactorCtrl, Reactor reactor) {
      this.reactor = reactor;
      this.reactorCtrl = reactorCtrl;
      timerUpdateSignal = reactor.newSignal();
      timerUpdateSequence = 0;
      controlInterfaceSignal = reactor.newSignal();
      controlInterfaceSignal.subscribe(this);
    }

    /*
     * Start the stopwatch timer (thread safe).
     */
    public void timerStart() {
      controlInterfaceSignal.signal(CommandIds.TIMER_START);
    }

    /*
     * Stop the stopwatch timer (thread safe).
     */
    public void timerStop() {
      controlInterfaceSignal.signal(CommandIds.TIMER_STOP);
    }

    /*
     * Reset the stopwatch timer (thread safe).
     */
    public void timerReset() {
      controlInterfaceSignal.signal(CommandIds.TIMER_RESET);
    }

    /*
     * Exit the stopwatch timer application (thread safe).
     */
    public void timerQuit() {
      controlInterfaceSignal.signal(CommandIds.TIMER_QUIT);
    }

    /*
     * Get a handle on the timer update signal.
     */
    public Signal<Integer> getTimerUpdateSignal() {
      return timerUpdateSignal;
    }

    /*
     * Access the current timer state.
     */
    public int getTimerData() {
      return timerValue;
    }

    /*
     * Increment the timer counter on each callback and signal to the view
     * components that the data has changed.
     */
    public void onTick(Integer data) {
      timerValue += 250;
      timerUpdateSignal.signal(new Integer(timerUpdateSequence++));
    }

    /*
     * Perform the control actions in the context of the reactor thread.
     */
    public void onSignal(Signal<CommandIds> signal, CommandIds commandId) {
      if (commandId == null) {
        System.out.println("Control interface shut down.");
      } else {
        switch (commandId) {

        // Start the timer.
        case TIMER_START:
          if (!timerRunning) {
            System.out.println("Starting the stopwatch timer.");
            reactor.runTimerRepeating(this, 250, 250, null);
            timerRunning = true;
          }
          break;

        // Stop the timer.
        case TIMER_STOP:
          if (timerRunning) {
            System.out.println("Stopping the stopwatch timer.");
            reactor.cancelTimer(this);
            timerRunning = false;
          }
          break;

        // Reset the timer.
        case TIMER_RESET:
          if (timerRunning) {
            System.out.println("Stopping the stopwatch timer.");
            reactor.cancelTimer(this);
            timerRunning = false;
          }
          System.out.println("Resetting the stopwatch timer.");
          timerValue = 0;
          timerUpdateSignal.signal(new Integer(timerUpdateSequence++));
          break;

        // Exit the timer stopwatch application.
        case TIMER_QUIT:
          System.out.println("Quitting the stopwatch application.");
          timerUpdateSignal.signalFinalize(null);
          controlInterfaceSignal.signalFinalize(null);
          reactorCtrl.stop();
          break;
        }
      }
    }
  }

  /**
   * Provide the Swing GUI view and controller component. This component plays
   * the role of both controller and viewer within the MVC framework.
   */
  private static class ViewerController implements Signalable<Integer> {

    // Set the size of the viewer frame.
    private static final int FRAME_WIDTH = 320;
    private static final int FRAME_HEIGHT = 200;

    // Local handle on the viewer frame and content panel.
    private JFrame frame;
    private JPanel panel;

    // Local handle on the application control interface.
    private ApplicationControl appControl;
    private ApplicationView appView;

    // Local version of the current stopwatch time.
    private volatile String currentTime = "UNKNOWN";

    /*
     * Start up the GUI component once the application service is available.
     */
    public void startup(ApplicationModel app) {

      // Cache references to the application's control and view
      // interfaces.
      appControl = app;
      appView = app;

      // Create the basic frame, wiring up the window close event to the
      // application shutdown control method.
      frame = new JFrame();
      frame.setSize(FRAME_WIDTH, FRAME_HEIGHT);
      frame.setTitle("Reaction Stopwatch Example");
      frame.addWindowListener(new WindowAdapter() {
        @Override
        public void windowClosing(WindowEvent e) {
          appControl.timerQuit();
        }
      });

      // Create a panel with a dedicated 'paintComponent' method which
      // prints the current stopwatch time.
      panel = new JPanel() {
        private static final long serialVersionUID = -1208586769373309418L;
        private final Font f = new Font("Sans", Font.BOLD, 24);

        @Override
        public void paintComponent(Graphics g) {
          super.paintComponent(g);
          Graphics2D g2 = (Graphics2D) g;
          g2.setFont(f);
          Rectangle2D bounds = f.getStringBounds(currentTime,
              g2.getFontRenderContext());
          g2.drawString(currentTime,
              (int) (getWidth() - bounds.getWidth()) / 2,
              2 * (int) (getHeight() - bounds.getHeight()) / 3);
        }
      };

      // Populate the content panel with the control buttons.
      JButton startButton = new JButton("Start");
      panel.add(startButton);
      JButton stopButton = new JButton("Stop");
      panel.add(stopButton);
      JButton resetButton = new JButton("Reset");
      panel.add(resetButton);

      // Add action listeners to the control buttons and wire them up to
      // the control interface.
      startButton.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent event) {
          appControl.timerStart();
        }
      });
      stopButton.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent event) {
          appControl.timerStop();
        }
      });
      resetButton.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent event) {
          appControl.timerReset();
        }
      });

      // Add the panel to the basic frame.
      updateCurrentTime(0);
      frame.add(panel);
      frame.setVisible(true);

      // Register this viewer in order to get update callbacks.
      appView.getTimerUpdateSignal().subscribe(this);
    }

    /*
     * Receive application view updates via the signal callback.
     */
    public void onSignal(Signal<Integer> signal, Integer data) {
      if (data == null) {
        System.out.println("View interface shut down.");
      } else {
        int timeVal = appView.getTimerData();
        updateCurrentTime(timeVal);
      }
    }

    /*
     * Print out the current stopwatch time.
     */
    private void updateCurrentTime(int timeVal) {
      DecimalFormat formatter = new DecimalFormat("00");
      currentTime = "" + formatter.format(timeVal / 3600000) + ":"
          + formatter.format((timeVal % 3600000) / 60000) + ":"
          + formatter.format((timeVal % 60000) / 1000) + "."
          + formatter.format((timeVal % 1000) / 10);
      panel.repaint();
    }
  }
}
