/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.examples.signal;

import com.zynaptic.reaction.Reactor;
import com.zynaptic.reaction.Signal;
import com.zynaptic.reaction.Signalable;
import com.zynaptic.reaction.Timeable;
import com.zynaptic.reaction.core.ReactorControl;
import com.zynaptic.reaction.core.ReactorCore;
import com.zynaptic.reaction.util.FixedUpMonotonicClock;
import com.zynaptic.reaction.util.ReactorLogSystemOut;

/**
 * Example of signal subscribe and unsubscribe operations, following the
 * publisher subscriber pattern.
 */
public class SignalSubscribeExample {
  public static void main(String[] args) throws InterruptedException {

    // Obtain a handle on the reactor control interface.
    ReactorControl reactorCtrl = ReactorCore.getReactorControl();
    Reactor reactor = ReactorCore.getReactor();

    // Start the reactor using the specified monotonic clock and log
    // service.
    System.out.println("Starting up reactor.");
    reactorCtrl.start(new FixedUpMonotonicClock(), new ReactorLogSystemOut());

    // Instantiate a publisher.
    Publisher publisher = new PublisherImpl(reactor);

    // Create a first subscriber.
    Thread.sleep(500);
    Subscriber subscriber1 = new Subscriber("Subscriber 1");
    subscriber1.setup(publisher);

    // Wait for a bit and set up a second subscriber.
    Thread.sleep(2000);
    Subscriber subscriber2 = new Subscriber("Subscriber 2");
    subscriber2.setup(publisher);

    // Wait for a bit and unsubscribe first subscriber.
    Thread.sleep(2000);
    subscriber1.teardown();

    // Wait for a bit and unsubscribe second subscriber.
    Thread.sleep(2000);
    subscriber2.teardown();

    // Wait for a bit then shut down the reactor.
    Thread.sleep(2000);
    reactorCtrl.stop();
    reactorCtrl.join();
    System.exit(0);
  }

  /**
   * Define the publisher interface. This defines the API for the publisher
   * component.
   */
  public static interface Publisher {

    // Get a handle on the change notification signal.
    public Signal<Integer> getChangeNotificationSignal();

    // Read back the publisher data.
    public String getPublisherData();
  }

  /**
   * Implementation of the publisher component. This just implements a counter
   * which increments every second.
   */
  public static class PublisherImpl implements Publisher, Timeable<Integer> {
    private Signal<Integer> changeNotificationSignal;
    private int counter = 0;
    private int sequence = 0;

    // Set up timer callbacks to run the counter.
    public PublisherImpl(Reactor reactor) {
      changeNotificationSignal = reactor.newSignal();
      reactor.runTimerRepeating(this, 1000, 1000, null);
    }

    // Get a handle on the change notification signal.
    public Signal<Integer> getChangeNotificationSignal() {
      return changeNotificationSignal.makeRestricted();
    }

    // Return the current value of the published data.
    public String getPublisherData() {
      return ("The counter value is now " + counter);
    }

    // Modify the published data on timer callback.
    public void onTick(Integer timerData) {
      counter += 1;
      changeNotificationSignal.signal(new Integer(sequence++));
    }
  }

  /**
   * Implementation of the subscriber component. This subscribes and
   * unsubscribes from the publisher.
   */
  public static class Subscriber implements Signalable<Integer> {
    private String name;
    private Publisher publisher;

    // Assign a name to the subscriber in the constructor.
    public Subscriber(String name) {
      this.name = name;
    }

    // The setup method is used to subscribe to the publisher.
    public void setup(Publisher publisher) {
      this.publisher = publisher;
      publisher.getChangeNotificationSignal().subscribe(this);
      System.out.println(name + " initial data : " + publisher.getPublisherData());
    }

    // The teardown method is used to unsubscribe from the publisher.
    public void teardown() {
      publisher.getChangeNotificationSignal().unsubscribe(this);
      System.out.println(name + " unsubscribed.");
    }

    // On change notifications, read back the publisher data and print it
    // out.
    public void onSignal(Signal<Integer> signal, Integer signalData) {
      System.out.println(name + " sequence " + signalData + " : " + publisher.getPublisherData());
    }
  }
}
