/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.examples.osgi;

import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.osgi.framework.ServiceReference;
import org.osgi.util.tracker.ServiceTracker;
import org.osgi.util.tracker.ServiceTrackerCustomizer;

import com.zynaptic.reaction.Reactor;
import com.zynaptic.reaction.Timeable;

/**
 * Bundle activation for the Reaction client example component.
 * 
 * @author Chris Holgate
 */
public class OsgiStartupExample implements BundleActivator,
    ServiceTrackerCustomizer<Reactor, Reactor> {

  // Local context and bundle tracker references.
  private BundleContext context;
  private ServiceTracker<Reactor, Reactor> tracker;

  // Reaction client lifecycle handler.
  private LifecycleHandler lifecycleHandler;

  /**
   * Implementation of the <code> BundleActivator.start </code> method which is
   * used to start up the example code on activation.
   * 
   * @param context
   *          The OSGi bundle context.
   */
  public synchronized void start(BundleContext context) {
    System.out.println("Starting up Reaction client bundle.");
    this.context = context;

    // Starting up the reactor service.
    tracker = new ServiceTracker<Reactor, Reactor>(context, Reactor.class.getName(), this);
    tracker.open();
  }

  /**
   * Implementation of the <code> ServiceTrackerCustomizer.addingService </code>
   * method which is used to indicate that the reactor service is available for
   * use by the client.
   * 
   * @param ref
   *          The service reference of the Reaction service.
   */
  public synchronized Reactor addingService(ServiceReference<Reactor> ref) {
    Reactor reactor = null;

    // Multiple reactor services are not expected.
    if (lifecycleHandler == null) {
      reactor = context.getService(ref);
      lifecycleHandler = new LifecycleHandler();
      System.out.println("Found reactor.");

      // Run the client lifecycle startup.
      lifecycleHandler.startup(reactor);
    }

    // Return the reactor as the active service object.
    return reactor;
  }

  /**
   * Implementation of the <code> BundleActivator.stop </code> method which is
   * used to stop the reactor example on deactivation.
   * 
   * @param context
   *          The OSGi bundle context.
   */
  public synchronized void stop(BundleContext context) {
    System.out.println("Halting Reaction client bundle.");

    // Perform orderly shutdown of the client before removing reactor
    // access.
    if (lifecycleHandler != null) {
      lifecycleHandler.teardown();
      lifecycleHandler = null;
    }
    tracker.close();
  }

  /**
   * Implementation of the
   * <code> ServiceTrackerCustomizer.removedService </code> method which is used
   * to indicate removal of the Reaction service. If called outside of an
   * ordered shutdown will abort the client due to loss of reactor
   * functionality.
   * 
   * @param ref
   *          The service reference of the Reaction service.
   * @param service
   *          The service object for the modified service.
   */
  public synchronized void removedService(ServiceReference<Reactor> ref, Reactor service) {
    System.out.println("Lost reactor.");

    // Abort the client service if it is still running.
    if (lifecycleHandler != null) {
      lifecycleHandler.abort();
      lifecycleHandler = null;
    }
  }

  /**
   * Implementation of the
   * <code> ServiceTrackerCustomizer.modifiedService </code> method which is
   * used to indicate changes to the properties of the attached service. Since
   * the Reaction service does not support dynamic property changes, this
   * callback should never be invoked.
   * 
   * @param ref
   *          The service reference of the Reaction service.
   * @param service
   *          The service object for the modified service.
   */
  public synchronized void modifiedService(ServiceReference<Reactor> ref, Reactor service) {
    System.out.println("Modifying reactor.");
  }

  /*
   * Example Reaction client lifecycle handler. This is an example of a standard
   * way of managing the lifecycle of Reaction clients.
   */
  private class LifecycleHandler {
    private Reactor reactor = null;
    private LivenessIndicator livenessIndicator = null;

    /*
     * Initialise the Reaction client on startup.
     */
    public void startup(Reactor reactor) {
      System.out.println("Running client startup handler.");
      livenessIndicator = new LivenessIndicator();
      reactor.runTimerRepeating(livenessIndicator, 1000, 1000, null);
      this.reactor = reactor;
    }

    /*
     * Tear down the Reaction client on orderly shutdown.
     */
    public void teardown() {
      System.out.println("Performing orderly client shutdown.");
      reactor.cancelTimer(livenessIndicator);
    }

    /*
     * Abort the Reaction client on loss of reactor service.
     */
    public void abort() {
      System.out.println("Aborting client on loss of reactor.");
    }

    /*
     * Simple timer callback handler for indicating client liveness.
     */
    private class LivenessIndicator implements Timeable<Integer> {
      private int callbackCount = 0;

      public void onTick(Integer data) {
        System.out.println("Reaction client is alive (" + (callbackCount++) + ")");
      }
    }
  }
}
