/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2013, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.examples.javafx;

import java.text.DecimalFormat;

import javafx.application.Application;
import javafx.application.Platform;
import javafx.beans.property.SimpleStringProperty;
import javafx.event.ActionEvent;
import javafx.event.EventHandler;
import javafx.scene.Scene;
import javafx.scene.SceneBuilder;
import javafx.scene.control.ButtonBuilder;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.BorderPaneBuilder;
import javafx.scene.layout.HBox;
import javafx.scene.layout.HBoxBuilder;
import javafx.scene.layout.StackPane;
import javafx.scene.layout.StackPaneBuilder;
import javafx.scene.text.Text;
import javafx.scene.text.TextBuilder;
import javafx.stage.Stage;
import javafx.stage.WindowEvent;

import com.zynaptic.reaction.Reactor;
import com.zynaptic.reaction.Signal;
import com.zynaptic.reaction.Signalable;
import com.zynaptic.reaction.Timeable;
import com.zynaptic.reaction.core.ReactorControl;
import com.zynaptic.reaction.core.ReactorCore;
import com.zynaptic.reaction.util.JavaMonotonicClock;
import com.zynaptic.reaction.util.ReactorLogSystemOut;

/**
 * Example of using the Reaction framework in conjunction with a GUI framework
 * (JavaFX) via the MVC pattern.
 * 
 * @author Chris Holgate
 */
public class JavaFxGuiExample {
  public static void main(String[] args) throws InterruptedException {

    // Start up the reactor.
    Reactor reactor = ReactorCore.getReactor();
    ReactorControl reactorCtrl = ReactorCore.getReactorControl();
    reactorCtrl.start(new JavaMonotonicClock(), new ReactorLogSystemOut());

    // Start up the application.
    ApplicationModel app = new ApplicationModel();
    app.startup(reactorCtrl, reactor);

    // Provide the control and viewer interfaces.
    appControl = app;
    appView = app;

    // Run the main JavaFx GUI polling loop.
    Application.launch(ViewerController.class, new String[0]);

    // Wait for the application to exit.
    reactorCtrl.join();
    System.exit(0);
  }

  /**
   * This is the interface which should be used for application control.
   */
  private static interface ApplicationControl {

    /**
     * Command interface used to start the stopwatch timer.
     */
    public void timerStart();

    /**
     * Command interface used to stop the stopwatch timer.
     */
    public void timerStop();

    /**
     * Command interface used to reset the stopwatch timer.
     */
    public void timerReset();

    /**
     * Command interface used to quit the stopwatch timer application.
     */
    public void timerQuit();

  }

  /**
   * This is the interface which should be used for viewer updates.
   */
  private static interface ApplicationView {

    /**
     * Get a handle on the timer update signal.
     */
    public Signal<Integer> getTimerUpdateSignal();

    /**
     * Read back the current timer value (in milliseconds).
     */
    public int getTimerData();

  }

  /**
   * Specify command IDs used to perform control actions.
   */
  enum CommandIds {
    TIMER_START, TIMER_STOP, TIMER_RESET, TIMER_QUIT
  };

  /**
   * Implement the application logic as a Reaction service.
   */
  private static class ApplicationModel implements Timeable<Integer>,
      Signalable<CommandIds>, ApplicationControl, ApplicationView {

    // Cached local reference to the reactor service.
    private Reactor reactor;
    private ReactorControl reactorCtrl;

    // Track the current timer state.
    private boolean timerRunning = false;
    private int timerValue = 0;

    // Local handle on the timer update signal.
    private Signal<Integer> timerUpdateSignal = null;
    private int timerUpdateSequence;

    // Local handle on the control interface signal.
    private Signal<CommandIds> controlInterfaceSignal = null;

    /*
     * Start up the application component once the Reaction service is
     * available.
     */
    public void startup(ReactorControl reactorCtrl, Reactor reactor) {
      this.reactor = reactor;
      this.reactorCtrl = reactorCtrl;
      timerUpdateSignal = reactor.newSignal();
      timerUpdateSequence = 0;
      controlInterfaceSignal = reactor.newSignal();
      controlInterfaceSignal.subscribe(this);
    }

    /*
     * Start the stopwatch timer (thread safe).
     */
    public void timerStart() {
      controlInterfaceSignal.signal(CommandIds.TIMER_START);
    }

    /*
     * Stop the stopwatch timer (thread safe).
     */
    public void timerStop() {
      controlInterfaceSignal.signal(CommandIds.TIMER_STOP);
    }

    /*
     * Reset the stopwatch timer (thread safe).
     */
    public void timerReset() {
      controlInterfaceSignal.signal(CommandIds.TIMER_RESET);
    }

    /*
     * Exit the stopwatch timer application (thread safe).
     */
    public void timerQuit() {
      controlInterfaceSignal.signal(CommandIds.TIMER_QUIT);
    }

    /*
     * Get a handle on the timer update signal.
     */
    public Signal<Integer> getTimerUpdateSignal() {
      return timerUpdateSignal;
    }

    /*
     * Access the current timer state.
     */
    public int getTimerData() {
      return timerValue;
    }

    /*
     * Increment the timer counter on each callback and signal to the view
     * components that the data has changed.
     */
    public void onTick(Integer data) {
      timerValue += 250;
      timerUpdateSignal.signal(new Integer(timerUpdateSequence++));
    }

    /*
     * Perform the control actions in the context of the reactor thread.
     */
    public void onSignal(Signal<CommandIds> signal, CommandIds commandId) {
      if (commandId == null) {
        System.out.println("Control interface shut down.");
      } else {
        switch (commandId) {

        // Start the timer.
        case TIMER_START:
          if (!timerRunning) {
            System.out.println("Starting the stopwatch timer.");
            reactor.runTimerRepeating(this, 250, 250, null);
            timerRunning = true;
          }
          break;

        // Stop the timer.
        case TIMER_STOP:
          if (timerRunning) {
            System.out.println("Stopping the stopwatch timer.");
            reactor.cancelTimer(this);
            timerRunning = false;
          }
          break;

        // Reset the timer.
        case TIMER_RESET:
          if (timerRunning) {
            System.out.println("Stopping the stopwatch timer.");
            reactor.cancelTimer(this);
            timerRunning = false;
          }
          System.out.println("Resetting the stopwatch timer.");
          timerValue = 0;
          timerUpdateSignal.signal(new Integer(timerUpdateSequence++));
          break;

        // Exit the timer stopwatch application.
        case TIMER_QUIT:
          System.out.println("Quitting the stopwatch application.");
          timerUpdateSignal.signalFinalize(null);
          controlInterfaceSignal.signalFinalize(null);
          reactorCtrl.stop();
          break;
        }
      }
    }
  }

  /*
   * Provide local handle on the application control interface.
   */
  private static ApplicationControl appControl;

  /*
   * Provide local handle on the application view interface.
   */
  private static ApplicationView appView;

  /**
   * Provide the JavaFx GUI view and controller component. This component plays
   * the role of both controller and viewer within the MVC framework.
   */
  public static class ViewerController extends Application {

    // Set the size of the viewer frame.
    private static final int FRAME_WIDTH = 320;
    private static final int FRAME_HEIGHT = 200;

    /*
     * JavaFX application entry point.
     */
    @Override
    public void start(Stage stage) throws Exception {

      // Create an observable timer object and register it to receive timer
      // updates.
      ObservableTimer observableTimer = new ObservableTimer();
      appView.getTimerUpdateSignal().subscribe(observableTimer);

      // Create a pane to hold the current timer string and bind the string
      // value to the observable timer value.
      StackPane timerValuePane = StackPaneBuilder.create()
          .styleClass("timer-pane").build();
      Text timerValueText = TextBuilder.create().styleClass("timer-text")
          .build();
      timerValueText.textProperty().bind(
          observableTimer.getObservableTimerString());
      timerValuePane.getChildren().add(timerValueText);

      // Create the horizontal row of control buttons.
      HBox buttonBox = HBoxBuilder.create().styleClass("control-box").build();
      buttonBox.getChildren().add(
          ButtonBuilder.create().text("Start").styleClass("control-button")
              .minWidth(75).onAction(new EventHandler<ActionEvent>() {
                public void handle(ActionEvent event) {
                  appControl.timerStart();
                }
              }).build());
      buttonBox.getChildren().add(
          ButtonBuilder.create().text("Stop").styleClass("control-button")
              .minWidth(75).onAction(new EventHandler<ActionEvent>() {
                public void handle(ActionEvent event) {
                  appControl.timerStop();
                }
              }).build());
      buttonBox.getChildren().add(
          ButtonBuilder.create().text("Reset").styleClass("control-button")
              .minWidth(75).onAction(new EventHandler<ActionEvent>() {
                public void handle(ActionEvent event) {
                  appControl.timerReset();
                }
              }).build());

      // Handle window close events.
      stage.setOnCloseRequest(new EventHandler<WindowEvent>() {
        public void handle(WindowEvent windowEvent) {
          if (windowEvent.getEventType() == WindowEvent.WINDOW_CLOSE_REQUEST) {
            appControl.timerQuit();
          }
        }
      });

      // Create the border pane to hold the other UI elements.
      BorderPane borderPane = BorderPaneBuilder.create().center(timerValuePane)
          .bottom(buttonBox).build();

      // Create the scene and display the GUI.
      Scene scene = SceneBuilder.create().root(borderPane).width(FRAME_WIDTH)
          .height(FRAME_HEIGHT).build();
      scene.getStylesheets().add(
          JavaFxGuiExample.class.getResource("JavaFxGuiExample.css")
              .toExternalForm());

      // Prepare the stage for displaying the application scene.
      stage.setScene(scene);
      stage.setTitle("Reaction Stopwatch Example");
      stage.setResizable(false);
      stage.show();
    }
  }

  /**
   * Implement the observable timer value. This binds update notifications from
   * the model to the timer string view.
   */
  private static class ObservableTimer implements Signalable<Integer>, Runnable {

    // The formatted timer string to be presented via the GUI.
    private String timerString;

    // The observable string object which is used for binding the timer value
    // to the JavaFX GUI.
    private final SimpleStringProperty stringProperty = new SimpleStringProperty(
        "00:00:00.00");

    /*
     * Accesses the observable timer string property.
     */
    public SimpleStringProperty getObservableTimerString() {
      return stringProperty;
    }

    /*
     * Signal event callback to notify changes on the stopwatch timer value.
     * Updates the stored string value and then schedules an update to the
     * observable string value from within the JavaFX thread.
     */
    public synchronized void onSignal(Signal<Integer> signalId,
        Integer timerUpdateSequence) {
      if (timerUpdateSequence == null) {
        System.out.println("View interface shut down.");
      } else {
        int timeVal = appView.getTimerData();
        DecimalFormat formatter = new DecimalFormat("00");
        timerString = "" + formatter.format(timeVal / 3600000) + ":"
            + formatter.format((timeVal % 3600000) / 60000) + ":"
            + formatter.format((timeVal % 60000) / 1000) + "."
            + formatter.format((timeVal % 1000) / 10);
        Platform.runLater(this);
      }
    }

    /*
     * JavaFX callback on run later event. This is called from the JavaFX
     * thread. Updates the value held by the observable string.
     */
    public synchronized void run() {
      stringProperty.set(timerString);
    }
  }
}
