/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.examples.deferred;

import java.util.Random;
import java.util.List;
import java.util.Iterator;

import com.zynaptic.reaction.Reactor;
import com.zynaptic.reaction.Threadable;
import com.zynaptic.reaction.Deferred;
import com.zynaptic.reaction.Deferrable;
import com.zynaptic.reaction.DeferredConcentrator;
import com.zynaptic.reaction.core.ReactorControl;
import com.zynaptic.reaction.core.ReactorCore;
import com.zynaptic.reaction.util.FixedUpMonotonicClock;
import com.zynaptic.reaction.util.ReactorLogSystemOut;

/**
 * Example of the use of a deferred concentrator to implement thread farm
 * behaviour.
 */
public class DeferredConcentratorExample {
  public static void main(String[] args) throws InterruptedException {

    // Obtain a handle on the reactor control interface.
    reactorCtrl = ReactorCore.getReactorControl();
    reactor = ReactorCore.getReactor();

    // Start the reactor using the specified monotonic clock and log
    // service.
    System.out.println("Starting up reactor.");
    reactorCtrl.start(new FixedUpMonotonicClock(), new ReactorLogSystemOut());

    // Start prime factorisation for a set of random numbers.
    Random random = new Random();
    System.out.println("Performing prime factorisation on the following numbers :");
    for (int i = 0; i < nums.length; i++) {
      nums[i] = 1 + random.nextInt(100000000);
      System.out.println("  " + nums[i]);
    }

    // Create a deferred concentrator object to collect all the results.
    DeferredConcentrator<String> resultsConcentrator = reactor.newDeferredConcentrator();

    // Initiate the long running tasks and attach the deferred results to
    // the deferred result concentrator.
    for (int i = 0; i < nums.length; i++) {
      Deferred<String> deferredResult = reactor.runThread(new PrimeFactorCalculator(), new Integer(
          nums[i]));
      resultsConcentrator.addInputDeferred(deferredResult);
    }

    // Once all deferred results have been added to the concentrator, attach
    // the deferred output handler.
    resultsConcentrator.getOutputDeferred().addDeferrable(new ResultsProcessor(), true);

    // Wait for test completion.
    reactorCtrl.join();
    System.exit(0);
  }

  // Local handles on the reactor component.
  private static ReactorControl reactorCtrl = null;
  private static Reactor reactor = null;

  // List of numbers to be factorised.
  private static int[] nums = new int[5];

  /**
   * Deferred result handler for processing the output of the deferred
   * concentrator.
   */
  private static class ResultsProcessor implements Deferrable<List<String>, String> {

    /*
     * Handle successful test completion.
     */
    public String onCallback(Deferred<List<String>> deferred, List<String> data) {
      System.out.println("Prime factorisation results :");
      Iterator<String> listIter = data.iterator();
      for (int i = 0; i < data.size(); i++) {
        System.out.println("  " + nums[i] + " => " + listIter.next());
      }
      reactorCtrl.stop();
      return null;
    }

    /*
     * The errback should not be triggered during this test.
     */
    public String onErrback(Deferred<List<String>> deferred, Exception error) {
      System.out.println("Unexpected error : " + error.toString());
      reactorCtrl.stop();
      return null;
    }
  }

  /**
   * Simple implementation of direct prime factor extraction as a long running
   * threadable task.
   */
  private static class PrimeFactorCalculator implements Threadable<Integer, String> {
    public String run(Integer data) throws InterruptedException {

      // Input for prime factorisation is passed as the data parameter.
      int num = data.intValue();
      String result = "" + num + " = 1";
      int i = 2;
      while (num >= i) {

        // Perform single prime factor iteration.
        if (num % i == 0) {
          num = num / i;
          result = result + " * " + i;
        } else {
          i++;
        }
      }
      return result;
    }
  }
}
