/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.examples.deferred;

import com.zynaptic.reaction.Deferrable;
import com.zynaptic.reaction.Deferred;
import com.zynaptic.reaction.Reactor;
import com.zynaptic.reaction.Timeable;
import com.zynaptic.reaction.core.ReactorControl;
import com.zynaptic.reaction.core.ReactorCore;
import com.zynaptic.reaction.util.FixedUpMonotonicClock;
import com.zynaptic.reaction.util.ReactorLogSystemOut;

/**
 * Simple example of deferred errback chaining.
 */
public class DeferredChainExample3 {
  public static void main(String[] args) throws InterruptedException {

    // Obtain a handle on the reactor control interface.
    reactorCtrl = ReactorCore.getReactorControl();
    reactor = ReactorCore.getReactor();

    // Start the reactor using the specified monotonic clock and log
    // service.
    System.out.println("Starting up reactor.");
    reactorCtrl.start(new FixedUpMonotonicClock(), new ReactorLogSystemOut());

    // Start up a long running task.
    Deferred<Integer> deferredInteger = new LongRunningTask().runTask();

    // Attach callback to convert from integer to string.
    Deferred<String> deferredString = deferredInteger.addDeferrable(new TypeConversionDeferrable(),
        false);

    // Attach callback to report string result.
    deferredString.addDeferrable(new TerminalDeferrable(), true);

    // Wait for reactor to exit.
    Thread.sleep(2000);
    reactorCtrl.stop();
    reactorCtrl.join();
    System.exit(0);
  }

  // Local handles on the reactor component.
  private static ReactorControl reactorCtrl = null;
  private static Reactor reactor = null;

  /**
   * Implementation of a type conversion deferrable object.
   */
  public static class TypeConversionDeferrable implements Deferrable<Integer, String> {

    // Callbacks convert the integer input to a string output.
    public String onCallback(Deferred<Integer> deferred, Integer data) {
      System.out.println("Integer parameter => " + data);
      return new String("Value = " + data);
    }

    // Errbacks are unused in this example.
    public String onErrback(Deferred<Integer> deferred, Exception error) {
      return null;
    }
  }

  /**
   * Implementation of a terminal deferrable object.
   */
  public static class TerminalDeferrable implements Deferrable<String, String> {

    // Callbacks report the string parameter input.
    public String onCallback(Deferred<String> deferred, String data) {
      System.out.println("String parameter => " + data);
      return null;
    }

    // Errbacks are unused in this example.
    public String onErrback(Deferred<String> deferred, Exception error) {
      return null;
    }
  }

  /**
   * Implementation of a long running task using timed events.
   */
  public static class LongRunningTask implements Timeable<Integer> {
    private Deferred<Integer> deferred = null;

    // Method used to initiate the long running task.
    public Deferred<Integer> runTask() {
      reactor.runTimerOneShot(this, 1000, null);
      deferred = reactor.newDeferred();
      return deferred.makeRestricted();
    }

    // Timer event is used to trigger deferred callback.
    public void onTick(Integer data) {
      deferred.callback(new Integer(42));
    }
  }
}
