/*
 * Zynaptic Reaction - An asynchronous programming framework for Java.
 * 
 * Copyright (c) 2009-2012, Zynaptic Limited. All rights reserved. 
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 * 
 * This code is free software; you can redistribute it and/or modify it under
 * the terms of the GNU General Public License version 2 only, as published by
 * the Free Software Foundation. Zynaptic Limited designates this particular
 * file as subject to the "Classpath" exception as provided in the LICENSE
 * file that accompanied this code.
 * 
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 * 
 * You should have received a copy of the GNU General Public License version 2
 * along with this work; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please visit www.zynaptic.com or contact reaction@zynaptic.com if you need
 * additional information or have any questions.
 */

package com.zynaptic.reaction.examples.deferred;

import com.zynaptic.reaction.Deferrable;
import com.zynaptic.reaction.Deferred;
import com.zynaptic.reaction.Reactor;
import com.zynaptic.reaction.Timeable;
import com.zynaptic.reaction.core.ReactorControl;
import com.zynaptic.reaction.core.ReactorCore;
import com.zynaptic.reaction.util.FixedUpMonotonicClock;
import com.zynaptic.reaction.util.ReactorLogSystemOut;

/**
 * Simple example of deferred errback chaining.
 */
public class DeferredChainExample2 {
  public static void main(String[] args) throws InterruptedException {

    // Obtain a handle on the reactor control interface.
    reactorCtrl = ReactorCore.getReactorControl();
    reactor = ReactorCore.getReactor();

    // Start the reactor using the specified monotonic clock and log
    // service.
    System.out.println("Starting up reactor.");
    reactorCtrl.start(new FixedUpMonotonicClock(), new ReactorLogSystemOut());

    // Start up a long running task.
    Deferred<String> deferred = new LongRunningTask().runTask();

    // Attach callbacks to the long running task.
    deferred.addDeferrable(new SimpleDeferrable("C1"), false);
    deferred.addDeferrable(new SimpleDeferrable("C2"), false);
    deferred.addDeferrable(new SimpleDeferrable("C3"), false);
    deferred.addDeferrable(new SimpleDeferrable("C4"), true);

    // Wait for reactor to exit.
    Thread.sleep(2000);
    reactorCtrl.stop();
    reactorCtrl.join();
    System.exit(0);
  }

  // Local handles on the reactor component.
  private static ReactorControl reactorCtrl = null;
  private static Reactor reactor = null;

  /**
   * Implementation of a simple deferrable object.
   */
  public static class SimpleDeferrable implements Deferrable<String, String> {
    String name;

    // Constructor assigns name of deferrable.
    public SimpleDeferrable(String name) {
      this.name = name;
    }

    // Callbacks are ignored in this example.
    public String onCallback(Deferred<String> deferred, String data) {
      return null;
    }

    // Errbacks are propagated in this example.
    public String onErrback(Deferred<String> deferred, Exception error) throws Exception {
      System.out.println(name + " input = " + error);
      throw (error);
    }
  }

  /**
   * Implementation of a long running task using timed events.
   */
  public static class LongRunningTask implements Timeable<Integer> {
    private Deferred<String> deferred = null;

    // Method used to initiate the long running task.
    public Deferred<String> runTask() {
      reactor.runTimerOneShot(this, 1000, null);
      deferred = reactor.newDeferred();
      return deferred.makeRestricted();
    }

    // Timer event is used to trigger deferred callback.
    public void onTick(Integer data) {
      deferred.errback(new Exception("Errback Exception"));
    }
  }
}
